<?php
/********************************************
 * SIMPLE CRUD FOR:
 * 1. waiting_period_rules
 * 2. policy_rules
 * 3. policy_waiting_rule_map
 ********************************************/

// ====== DB CONFIG ======
include("database.php");

/*
$con = new mysqli($DB_HOST, $DB_USER, $DB_PASS, $DB_NAME);
if ($con->connect_error) {
    die('Database connection failed: ' . $con->connect_error);
}
$con->set_charset('utf8mb4');
*/
// ====== SMALL HELPERS ======
function post($key, $default = '')
{
    return isset($_POST[$key]) ? trim($_POST[$key]) : $default;
}

$activeTab = $_GET['tab'] ?? 'policy_rules'; // default tab

// ====== HANDLE POST (CREATE / UPDATE / DELETE) ======
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $entity = post('entity');
    $action = post('action');
    $msg    = '';
    $ok     = false;

    if ($entity === 'policy_rules') {
        // TABLE: policy_rules(rule_id, rule_code, rule_category, rule_description, is_active)
        if ($action === 'create') {
            $rule_code        = post('rule_code');
            $rule_category    = post('rule_category');
            $rule_description = post('rule_description');
            $is_active        = post('is_active') === '1' ? 1 : 0;

            $stmt = $con->prepare("
                INSERT INTO policy_rules (rule_code, rule_category, rule_description, is_active)
                VALUES (?, ?, ?, ?)
            ");
            if ($stmt) {
                $stmt->bind_param("sssi", $rule_code, $rule_category, $rule_description, $is_active);
                $ok = $stmt->execute();
                $msg = $ok ? 'Policy rule added successfully.' : 'Failed to add policy rule.';
                $stmt->close();
            } else {
                $msg = 'Prepare failed: ' . $con->error;
            }

        } elseif ($action === 'update') {
            $rule_id          = (int) post('rule_id');
            $rule_code        = post('rule_code');
            $rule_category    = post('rule_category');
            $rule_description = post('rule_description');
            $is_active        = post('is_active') === '1' ? 1 : 0;

            $stmt = $con->prepare("
                UPDATE policy_rules
                SET rule_code = ?, rule_category = ?, rule_description = ?, is_active = ?
                WHERE rule_id = ?
            ");
            if ($stmt) {
                $stmt->bind_param("sssii", $rule_code, $rule_category, $rule_description, $is_active, $rule_id);
                $ok = $stmt->execute();
                $msg = $ok ? 'Policy rule updated successfully.' : 'Failed to update policy rule.';
                $stmt->close();
            } else {
                $msg = 'Prepare failed: ' . $con->error;
            }

        } elseif ($action === 'delete') {
            $rule_id = (int) post('rule_id');

            $stmt = $con->prepare("DELETE FROM policy_rules WHERE rule_id = ?");
            if ($stmt) {
                $stmt->bind_param("i", $rule_id);
                $ok = $stmt->execute();
                $msg = $ok ? 'Policy rule deleted successfully.' : 'Failed to delete policy rule.';
                $stmt->close();
            } else {
                $msg = 'Prepare failed: ' . $con->error;
            }
        }

        $activeTab = 'policy_rules';
    }

    if ($entity === 'waiting_period_rules') {
        // TABLE: waiting_period_rules(rule_id, life_type, min_age, max_age, waiting_period_months, rule_description)
        if ($action === 'create') {
            $life_type            = post('life_type');
            $min_age              = (int) post('min_age');
            $max_age              = (int) post('max_age');
            $waiting_period_months= (int) post('waiting_period_months');
            $rule_description     = post('rule_description');

            $stmt = $con->prepare("
                INSERT INTO waiting_period_rules (life_type, min_age, max_age, waiting_period_months, rule_description)
                VALUES (?, ?, ?, ?, ?)
            ");
            if ($stmt) {
                $stmt->bind_param("siiis", $life_type, $min_age, $max_age, $waiting_period_months, $rule_description);
                $ok = $stmt->execute();
                $msg = $ok ? 'Waiting period rule added successfully.' : 'Failed to add waiting period rule.';
                $stmt->close();
            } else {
                $msg = 'Prepare failed: ' . $con->error;
            }

        } elseif ($action === 'update') {
            $rule_id              = (int) post('rule_id');
            $life_type            = post('life_type');
            $min_age              = (int) post('min_age');
            $max_age              = (int) post('max_age');
            $waiting_period_months= (int) post('waiting_period_months');
            $rule_description     = post('rule_description');

            $stmt = $con->prepare("
                UPDATE waiting_period_rules
                SET life_type = ?, min_age = ?, max_age = ?, waiting_period_months = ?, rule_description = ?
                WHERE rule_id = ?
            ");
            if ($stmt) {
                $stmt->bind_param("siiisi", $life_type, $min_age, $max_age, $waiting_period_months, $rule_description, $rule_id);
                $ok = $stmt->execute();
                $msg = $ok ? 'Waiting period rule updated successfully.' : 'Failed to update waiting period rule.';
                $stmt->close();
            } else {
                $msg = 'Prepare failed: ' . $con->error;
            }

        } elseif ($action === 'delete') {
            $rule_id = (int) post('rule_id');

            $stmt = $con->prepare("DELETE FROM waiting_period_rules WHERE rule_id = ?");
            if ($stmt) {
                $stmt->bind_param("i", $rule_id);
                $ok = $stmt->execute();
                $msg = $ok ? 'Waiting period rule deleted successfully.' : 'Failed to delete waiting period rule.';
                $stmt->close();
            } else {
                $msg = 'Prepare failed: ' . $con->error;
            }
        }

        $activeTab = 'waiting_period_rules';
    }

    if ($entity === 'policy_waiting_rule_map') {
        // TABLE: policy_waiting_rule_map(id, policy_rule_id, waiting_period_rule_id)
        if ($action === 'create') {
            $policy_rule_id       = (int) post('policy_rule_id');
            $waiting_period_rule_id = (int) post('waiting_period_rule_id');

            $stmt = $con->prepare("
                INSERT INTO policy_waiting_rule_map (policy_rule_id, waiting_period_rule_id)
                VALUES (?, ?)
            ");
            if ($stmt) {
                $stmt->bind_param("ii", $policy_rule_id, $waiting_period_rule_id);
                $ok = $stmt->execute();
                $msg = $ok ? 'Map record added successfully.' : 'Failed to add map record.';
                $stmt->close();
            } else {
                $msg = 'Prepare failed: ' . $con->error;
            }

        } elseif ($action === 'update') {
            $id                   = (int) post('id');
            $policy_rule_id       = (int) post('policy_rule_id');
            $waiting_period_rule_id = (int) post('waiting_period_rule_id');

            $stmt = $con->prepare("
                UPDATE policy_waiting_rule_map
                SET policy_rule_id = ?, waiting_period_rule_id = ?
                WHERE id = ?
            ");
            if ($stmt) {
                $stmt->bind_param("iii", $policy_rule_id, $waiting_period_rule_id, $id);
                $ok = $stmt->execute();
                $msg = $ok ? 'Map record updated successfully.' : 'Failed to update map record.';
                $stmt->close();
            } else {
                $msg = 'Prepare failed: ' . $con->error;
            }

        } elseif ($action === 'delete') {
            $id = (int) post('id');

            $stmt = $con->prepare("DELETE FROM policy_waiting_rule_map WHERE id = ?");
            if ($stmt) {
                $stmt->bind_param("i", $id);
                $ok = $stmt->execute();
                $msg = $ok ? 'Map record deleted successfully.' : 'Failed to delete map record.';
                $stmt->close();
            } else {
                $msg = 'Prepare failed: ' . $con->error;
            }
        }

        $activeTab = 'policy_waiting_rule_map';
    }

    // Redirect to avoid resubmission
    header("Location: " . $_SERVER['PHP_SELF'] . "?tab={$activeTab}&msg=" . urlencode($msg));
    exit;
}

// ====== FETCH DATA FOR DISPLAY / EDIT ======

// Policy rules list
$policyRules = [];
$res = $con->query("SELECT * FROM policy_rules ORDER BY rule_id ASC");
if ($res) {
    while ($row = $res->fetch_assoc()) {
        $policyRules[] = $row;
    }
    $res->free();
}

// Waiting period rules list
$waitingRules = [];
$res = $con->query("SELECT * FROM waiting_period_rules ORDER BY rule_id ASC");
if ($res) {
    while ($row = $res->fetch_assoc()) {
        $waitingRules[] = $row;
    }
    $res->free();
}

// Map list, with joins for nicer display
$maps = [];
$sql = "
    SELECT m.id,
           m.policy_rule_id,
           pr.rule_code,
           m.waiting_period_rule_id,
           wr.life_type,
           wr.min_age,
           wr.max_age
    FROM policy_waiting_rule_map m
    LEFT JOIN policy_rules pr ON m.policy_rule_id = pr.rule_id
    LEFT JOIN waiting_period_rules wr ON m.waiting_period_rule_id = wr.rule_id
    ORDER BY m.id ASC
";
$res = $con->query($sql);
if ($res) {
    while ($row = $res->fetch_assoc()) {
        $maps[] = $row;
    }
    $res->free();
}

// ====== EDIT RECORDS (IF ANY) ======
$editPolicyRule = null;
if ($activeTab === 'policy_rules' && isset($_GET['edit_id'])) {
    $edit_id = (int) $_GET['edit_id'];
    $stmt = $con->prepare("SELECT * FROM policy_rules WHERE rule_id = ?");
    if ($stmt) {
        $stmt->bind_param("i", $edit_id);
        $stmt->execute();
        $res = $stmt->get_result();
        $editPolicyRule = $res->fetch_assoc();
        $stmt->close();
    }
}

$editWaitingRule = null;
if ($activeTab === 'waiting_period_rules' && isset($_GET['edit_id'])) {
    $edit_id = (int) $_GET['edit_id'];
    $stmt = $con->prepare("SELECT * FROM waiting_period_rules WHERE rule_id = ?");
    if ($stmt) {
        $stmt->bind_param("i", $edit_id);
        $stmt->execute();
        $res = $stmt->get_result();
        $editWaitingRule = $res->fetch_assoc();
        $stmt->close();
    }
}

$editMap = null;
if ($activeTab === 'policy_waiting_rule_map' && isset($_GET['edit_id'])) {
    $edit_id = (int) $_GET['edit_id'];
    $stmt = $con->prepare("SELECT * FROM policy_waiting_rule_map WHERE id = ?");
    if ($stmt) {
        $stmt->bind_param("i", $edit_id);
        $stmt->execute();
        $res = $stmt->get_result();
        $editMap = $res->fetch_assoc();
        $stmt->close();
    }
}

$msg = $_GET['msg'] ?? '';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Policy & Waiting Rules Manager</title>
    <!-- Bootstrap 4 CDN -->
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
    <style>
        body {
            background: #f5f5f5;
            font-size: 0.9rem;
        }
        .content {
            margin-top: 30px;
        }
        .card-header-primary {
            background: #6ec1e4;
            color: white;
            text-align: center;
            padding: 15px;
        }
        .card-title {
            color: rgb(10, 44, 234);
            font-weight: bold;
            text-align: center;
        }
        .table-sm td, .table-sm th {
            padding: .3rem;
        }
        textarea {
            resize: vertical;
        }
        .nav-pills .nav-link.active {
            background-color: #6ec1e4;
        }
        @media (max-width: 768px) {
            .col-md-8, .col-md-12 {
                padding-left: 5px;
                padding-right: 5px;
            }
        }
    </style>    
</head>
<body>

<?php include 'main_menu.php'; ?>

<style>
        body {
    background-image: url('images/ayo_banner4.png'); /* Replace with your actual image path */
    background-size: cover;
  background-position: center;
    background-repeat: no-repeat;
    background-attachment: fixed;
    opacity: 0.9; /* Adjust transparency (0.1 to 1) */
}

body::before {
    content: "";
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-image: url('images/ayo_banner4.png');
    background-size: cover;
    background-position: center;
    background-repeat: no-repeat;
    background-attachment: fixed;
    opacity: 0.5; /* Adjust transparency */
    z-index: -1;
}
</style>
<div class="content">
    <div class="container-fluid">
        <?php if ($msg): ?>
            <div class="row">
                <div class="col-md-12">
                    <div class="alert alert-info alert-dismissible fade show" role="alert">
                        <?php echo htmlspecialchars($msg); ?>
                        <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                            <span aria-hidden="true">&times;</span>
                        </button>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <div class="row">
            <!-- Main Form/Card Section -->
            <div class="col-md-12">
                <div class="card">
                    <!-- TEMPLATE A HEADER -->
                    <div class="card-header card-header-primary" style="background: #6ec1e4; color: white; text-align: center; padding: 15px;">
                        <h4 class="card-title">Add / Update Product Rules</h4>
                        <p class="card-category">Please complete the form below</p>
                    </div>

                    <div class="card-body">
                        <!-- NAV PILLS FOR TABS -->
                        <ul class="nav nav-pills mb-3" id="pills-tab" role="tablist">
                            <li class="nav-item">
                                <a class="nav-link <?php echo $activeTab === 'policy_rules' ? 'active' : ''; ?>"
                                   href="?tab=policy_rules">Policy Rules</a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link <?php echo $activeTab === 'waiting_period_rules' ? 'active' : ''; ?>"
                                   href="?tab=waiting_period_rules">Waiting Period Rules</a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link <?php echo $activeTab === 'policy_waiting_rule_map' ? 'active' : ''; ?>"
                                   href="?tab=policy_waiting_rule_map">Policy & Waiting Map</a>
                            </li>
                        </ul>

                        <div class="tab-content">
                            <!-- POLICY RULES TAB -->
                            <?php
                            $isEdit = $editPolicyRule !== null;
                            ?>
                            <div class="tab-pane fade <?php echo $activeTab === 'policy_rules' ? 'show active' : ''; ?>">
                                <h5 class="mb-3"><?php echo $isEdit ? 'Update Policy Rule' : 'Add New Policy Rule'; ?></h5>
                                <form method="post" class="mb-4">
                                    <input type="hidden" name="entity" value="policy_rules">
                                    <input type="hidden" name="action" value="<?php echo $isEdit ? 'update' : 'create'; ?>">
                                    <?php if ($isEdit): ?>
                                        <input type="hidden" name="rule_id"
                                               value="<?php echo (int)$editPolicyRule['rule_id']; ?>">
                                    <?php endif; ?>

                                    <div class="form-row">
                                        <div class="form-group col-md-4">
                                            <label>Rule Code</label>
                                            <input type="text" name="rule_code" class="form-control"
                                                   required
                                                   value="<?php echo $isEdit ? htmlspecialchars($editPolicyRule['rule_code']) : ''; ?>">
                                        </div>
                                        <div class="form-group col-md-4">
                                            <label>Category</label>
                                            <input type="text" name="rule_category" class="form-control"
                                                   required
                                                   value="<?php echo $isEdit ? htmlspecialchars($editPolicyRule['rule_category']) : ''; ?>">
                                        </div>
                                        <div class="form-group col-md-4">
                                            <label>Is Active</label>
                                            <select name="is_active" class="form-control">
                                                <?php
                                                $currentActive = $isEdit ? (int)$editPolicyRule['is_active'] : 1;
                                                ?>
                                                <option value="1" <?php echo $currentActive === 1 ? 'selected' : ''; ?>>Active</option>
                                                <option value="0" <?php echo $currentActive === 0 ? 'selected' : ''; ?>>Inactive</option>
                                            </select>
                                        </div>
                                    </div>

                                    <div class="form-group">
                                        <label>Rule Description</label>
                                        <textarea name="rule_description" class="form-control" rows="3" required><?php
                                            echo $isEdit ? htmlspecialchars($editPolicyRule['rule_description']) : '';
                                        ?></textarea>
                                    </div>

                                    <button type="submit" class="btn btn-primary">
                                        <?php echo $isEdit ? 'Update Rule' : 'Save Rule'; ?>
                                    </button>
                                    <?php if ($isEdit): ?>
                                        <a href="?tab=policy_rules" class="btn btn-secondary btn-sm">Cancel Edit</a>
                                    <?php endif; ?>
                                </form>

                                <!-- LIST TABLE -->
                                <h6>Existing Policy Rules</h6>
                                <div class="table-responsive">
                                    <table class="table table-bordered table-striped table-sm">
                                        <thead class="thead-light">
                                        <tr>
                                            <th>ID</th>
                                            <th>Code</th>
                                            <th>Category</th>
                                            <th>Description</th>
                                            <th>Active</th>
                                            <th style="width:120px;">Actions</th>
                                        </tr>
                                        </thead>
                                        <tbody>
                                        <?php if (count($policyRules) === 0): ?>
                                            <tr>
                                                <td colspan="6" class="text-center">No records found.</td>
                                            </tr>
                                        <?php else: ?>
                                            <?php foreach ($policyRules as $row): ?>
                                                <tr>
                                                    <td><?php echo (int)$row['rule_id']; ?></td>
                                                    <td><?php echo htmlspecialchars($row['rule_code']); ?></td>
                                                    <td><?php echo htmlspecialchars($row['rule_category']); ?></td>
                                                    <td><?php echo htmlspecialchars($row['rule_description']); ?></td>
                                                    <td><?php echo (int)$row['is_active'] === 1 ? 'Yes' : 'No'; ?></td>
                                                    <td>
                                                        <a href="?tab=policy_rules&edit_id=<?php echo (int)$row['rule_id']; ?>"
                                                           class="btn btn-sm btn-info">Edit</a>
                                                        <form method="post" style="display:inline-block;"
                                                              onsubmit="return confirm('Delete this rule?');">
                                                            <input type="hidden" name="entity" value="policy_rules">
                                                            <input type="hidden" name="action" value="delete">
                                                            <input type="hidden" name="rule_id"
                                                                   value="<?php echo (int)$row['rule_id']; ?>">
                                                            <button type="submit" class="btn btn-sm btn-danger">Del</button>
                                                        </form>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>

                            <!-- WAITING PERIOD RULES TAB -->
                            <?php $isEditW = $editWaitingRule !== null; ?>
                            <div class="tab-pane fade <?php echo $activeTab === 'waiting_period_rules' ? 'show active' : ''; ?>">
                                <h5 class="mb-3"><?php echo $isEditW ? 'Update Waiting Period Rule' : 'Add New Waiting Period Rule'; ?></h5>
                                <form method="post" class="mb-4">
                                    <input type="hidden" name="entity" value="waiting_period_rules">
                                    <input type="hidden" name="action" value="<?php echo $isEditW ? 'update' : 'create'; ?>">
                                    <?php if ($isEditW): ?>
                                        <input type="hidden" name="rule_id"
                                               value="<?php echo (int)$editWaitingRule['rule_id']; ?>">
                                    <?php endif; ?>

                                    <div class="form-row">
                                        <div class="form-group col-md-4">
                                            <label>Life Type</label>
                                            <input type="text" name="life_type" class="form-control" required
                                                   value="<?php echo $isEditW ? htmlspecialchars($editWaitingRule['life_type']) : ''; ?>">
                                        </div>
                                        <div class="form-group col-md-2">
                                            <label>Min Age</label>
                                            <input type="number" name="min_age" class="form-control" required
                                                   value="<?php echo $isEditW ? (int)$editWaitingRule['min_age'] : ''; ?>">
                                        </div>
                                        <div class="form-group col-md-2">
                                            <label>Max Age</label>
                                            <input type="number" name="max_age" class="form-control" required
                                                   value="<?php echo $isEditW ? (int)$editWaitingRule['max_age'] : ''; ?>">
                                        </div>
                                        <div class="form-group col-md-4">
                                            <label>Waiting Period (Months)</label>
                                            <input type="number" name="waiting_period_months" class="form-control" required
                                                   value="<?php echo $isEditW ? (int)$editWaitingRule['waiting_period_months'] : ''; ?>">
                                        </div>
                                    </div>

                                    <div class="form-group">
                                        <label>Rule Description</label>
                                        <textarea name="rule_description" class="form-control" rows="3" required><?php
                                            echo $isEditW ? htmlspecialchars($editWaitingRule['rule_description']) : '';
                                        ?></textarea>
                                    </div>

                                    <button type="submit" class="btn btn-primary">
                                        <?php echo $isEditW ? 'Update Rule' : 'Save Rule'; ?>
                                    </button>
                                    <?php if ($isEditW): ?>
                                        <a href="?tab=waiting_period_rules" class="btn btn-secondary btn-sm">Cancel Edit</a>
                                    <?php endif; ?>
                                </form>

                                <!-- LIST TABLE -->
                                <h6>Existing Waiting Period Rules</h6>
                                <div class="table-responsive">
                                    <table class="table table-bordered table-striped table-sm">
                                        <thead class="thead-light">
                                        <tr>
                                            <th>ID</th>
                                            <th>Life Type</th>
                                            <th>Age Range</th>
                                            <th>Wait (Months)</th>
                                            <th>Description</th>
                                            <th style="width:120px;">Actions</th>
                                        </tr>
                                        </thead>
                                        <tbody>
                                        <?php if (count($waitingRules) === 0): ?>
                                            <tr>
                                                <td colspan="6" class="text-center">No records found.</td>
                                            </tr>
                                        <?php else: ?>
                                            <?php foreach ($waitingRules as $row): ?>
                                                <tr>
                                                    <td><?php echo (int)$row['rule_id']; ?></td>
                                                    <td><?php echo htmlspecialchars($row['life_type']); ?></td>
                                                    <td><?php echo (int)$row['min_age']; ?> - <?php echo (int)$row['max_age']; ?></td>
                                                    <td><?php echo (int)$row['waiting_period_months']; ?></td>
                                                    <td><?php echo htmlspecialchars($row['rule_description']); ?></td>
                                                    <td>
                                                        <a href="?tab=waiting_period_rules&edit_id=<?php echo (int)$row['rule_id']; ?>"
                                                           class="btn btn-sm btn-info">Edit</a>
                                                        <form method="post" style="display:inline-block;"
                                                              onsubmit="return confirm('Delete this waiting rule?');">
                                                            <input type="hidden" name="entity" value="waiting_period_rules">
                                                            <input type="hidden" name="action" value="delete">
                                                            <input type="hidden" name="rule_id"
                                                                   value="<?php echo (int)$row['rule_id']; ?>">
                                                            <button type="submit" class="btn btn-sm btn-danger">Del</button>
                                                        </form>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>

                            <!-- POLICY & WAITING MAP TAB -->
                            <?php $isEditM = $editMap !== null; ?>
                            <div class="tab-pane fade <?php echo $activeTab === 'policy_waiting_rule_map' ? 'show active' : ''; ?>">
                                <h5 class="mb-3"><?php echo $isEditM ? 'Update Policy & Waiting Map' : 'Add New Policy & Waiting Map'; ?></h5>
                                <form method="post" class="mb-4">
                                    <input type="hidden" name="entity" value="policy_waiting_rule_map">
                                    <input type="hidden" name="action" value="<?php echo $isEditM ? 'update' : 'create'; ?>">
                                    <?php if ($isEditM): ?>
                                        <input type="hidden" name="id" value="<?php echo (int)$editMap['id']; ?>">
                                    <?php endif; ?>

                                    <div class="form-row">
                                        <div class="form-group col-md-6">
                                            <label>Policy Rule</label>
                                            <select name="policy_rule_id" class="form-control" required>
                                                <option value="">-- Select Policy Rule --</option>
                                                <?php
                                                $currentPolicyId = $isEditM ? (int)$editMap['policy_rule_id'] : 0;
                                                foreach ($policyRules as $pr) {
                                                    $sel = $currentPolicyId === (int)$pr['rule_id'] ? 'selected' : '';
                                                    echo '<option value="' . (int)$pr['rule_id'] . '" ' . $sel . '>'
                                                        . (int)$pr['rule_id'] . ' - ' . htmlspecialchars($pr['rule_code'])
                                                        . '</option>';
                                                }
                                                ?>
                                            </select>
                                        </div>
                                        <div class="form-group col-md-6">
                                            <label>Waiting Period Rule</label>
                                            <select name="waiting_period_rule_id" class="form-control" required>
                                                <option value="">-- Select Waiting Rule --</option>
                                                <?php
                                                $currentWaitId = $isEditM ? (int)$editMap['waiting_period_rule_id'] : 0;
                                                foreach ($waitingRules as $wr) {
                                                    $sel = $currentWaitId === (int)$wr['rule_id'] ? 'selected' : '';
                                                    echo '<option value="' . (int)$wr['rule_id'] . '" ' . $sel . '>'
                                                        . (int)$wr['rule_id'] . ' - '
                                                        . htmlspecialchars($wr['life_type']) . ' ('
                                                        . (int)$wr['min_age'] . '-' . (int)$wr['max_age'] . ' yrs)'
                                                        . '</option>';
                                                }
                                                ?>
                                            </select>
                                        </div>
                                    </div>

                                    <button type="submit" class="btn btn-primary">
                                        <?php echo $isEditM ? 'Update Map' : 'Save Map'; ?>
                                    </button>
                                    <?php if ($isEditM): ?>
                                        <a href="?tab=policy_waiting_rule_map" class="btn btn-secondary btn-sm">Cancel Edit</a>
                                    <?php endif; ?>
                                </form>

                                <!-- LIST TABLE -->
                                <h6>Existing Policy & Waiting Maps</h6>
                                <div class="table-responsive">
                                    <table class="table table-bordered table-striped table-sm">
                                        <thead class="thead-light">
                                        <tr>
                                            <th>ID</th>
                                            <th>Policy Rule</th>
                                            <th>Waiting Rule</th>
                                            <th style="width:120px;">Actions</th>
                                        </tr>
                                        </thead>
                                        <tbody>
                                        <?php if (count($maps) === 0): ?>
                                            <tr>
                                                <td colspan="4" class="text-center">No records found.</td>
                                            </tr>
                                        <?php else: ?>
                                            <?php foreach ($maps as $row): ?>
                                                <tr>
                                                    <td><?php echo (int)$row['id']; ?></td>
                                                    <td>
                                                        <?php
                                                        echo (int)$row['policy_rule_id'] . ' - ' . htmlspecialchars($row['rule_code']);
                                                        ?>
                                                    </td>
                                                    <td>
                                                        <?php
                                                        echo (int)$row['waiting_period_rule_id'] . ' - '
                                                            . htmlspecialchars($row['life_type']) . ' ('
                                                            . (int)$row['min_age'] . '-' . (int)$row['max_age'] . ' yrs)';
                                                        ?>
                                                    </td>
                                                    <td>
                                                        <a href="?tab=policy_waiting_rule_map&edit_id=<?php echo (int)$row['id']; ?>"
                                                           class="btn btn-sm btn-info">Edit</a>
                                                        <form method="post" style="display:inline-block;"
                                                              onsubmit="return confirm('Delete this map record?');">
                                                            <input type="hidden" name="entity" value="policy_waiting_rule_map">
                                                            <input type="hidden" name="action" value="delete">
                                                            <input type="hidden" name="id"
                                                                   value="<?php echo (int)$row['id']; ?>">
                                                            <button type="submit" class="btn btn-sm btn-danger">Del</button>
                                                        </form>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div> <!-- /tab-content -->
                    </div> <!-- /card-body -->
                </div> <!-- /card -->
            </div> <!-- /col -->
        </div> <!-- /row -->
    </div> <!-- /container-fluid -->
</div> <!-- /content -->

<script src="https://code.jquery.com/jquery-3.5.1.slim.min.js"></script>
<script src="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/js/bootstrap.bundle.min.js"></script>
</body>
</html>
