<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header("Content-Type: application/json; charset=utf-8");

session_start();

include("database.php"); // must create $con (mysqli)

$path = $_SERVER['REQUEST_URI'] ?? 'script_A';
$userindex = $_SESSION['userindex'] ?? 0;

if (!isset($con) || !$con) {
    echo json_encode(["ok" => false, "error" => "Database connection not available"]);
    exit;
}

/* =========================
   LOG FUNCTION
========================= */
function log_activity($con, $userindex, $caption, $path, $data_inf) {
    $sql = "INSERT INTO users_logs (userindex, caption, logs_date, path, data_inf)
            VALUES (?, ?, NOW(), ?, ?)";

    if ($stmt = mysqli_prepare($con, $sql)) {
        mysqli_stmt_bind_param($stmt, "isss", $userindex, $caption, $path, $data_inf);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
    }
}

/* =========================
   METHOD CHECK
========================= */
if ($_SERVER["REQUEST_METHOD"] !== "POST") {
    log_activity(
        $con,
        $userindex,
        "INVALID REQUEST METHOD",
        $path,
        json_encode($_SERVER)
    );

    http_response_code(405);
    echo json_encode(["ok" => false, "error" => "Method not allowed"]);
    exit;
}

/* =========================
   READ INPUT
========================= */
$product_index  = $_POST["product_index"]  ?? null;
$sum_assured = $_POST["sum_assured"] ?? null;
$no          = $_POST["no"]          ?? null;

if ($product_index === null || $sum_assured === null) {
    $raw = file_get_contents("php://input");
    $data = json_decode($raw, true);
    if (is_array($data)) {
        $product_index  = $data["product_index"]  ?? $product_index;
        $sum_assured = $data["sum_assured"] ?? $sum_assured;
        $no          = $data["no"]          ?? $no;
    }
}

if ($no === null || $no === "") {
    $no = 0;
}

/* =========================
   VALIDATION
========================= */
if ($product_index === null || $sum_assured === null || $product_index === "" || $sum_assured === "") {
    log_activity(
        $con,
        $userindex,
        "VALIDATION FAILED",
        $path,
        json_encode($_REQUEST)
    );

    http_response_code(400);
    echo json_encode(["ok" => false, "error" => "Missing parameters"]);
    exit;
}

$product_index = (int)$product_index;
$no = (int)$no;
$sum_assured_str = number_format((float)$sum_assured, 2, '.', '');

if (!in_array($no, [0, 1, 2], true)) {
    log_activity(
        $con,
        $userindex,
        "INVALID NO VALUE",
        $path,
        json_encode(["no" => $no])
    );

    http_response_code(400);
    echo json_encode(["ok" => false, "error" => "Invalid no"]);
    exit;
}

/* =========================
   PREMIUM COLUMN
========================= */
if ($no === 0) {
    $premium_col = "premium_main_life";
} elseif ($no === 1) {
    $premium_col = "premium_main_plus_one_parent";
} else {
    $premium_col = "premium_main_plus_two_parents";
}

/* =========================
   DB QUERY
========================= */
$sql = "SELECT $premium_col AS premium, product_name
        FROM cover_levels_view
        WHERE product_index = ? AND sum_assured = ?
        LIMIT 1";

$stmt = mysqli_prepare($con, $sql);
if (!$stmt) {
    log_activity(
        $con,
        $userindex,
        "DB PREPARE FAILED",
        $path,
        mysqli_error($con)
    );

    echo json_encode(["ok" => false, "error" => "Prepare failed"]);
    exit;
}

mysqli_stmt_bind_param($stmt, "is", $product_index, $sum_assured_str);

if (!mysqli_stmt_execute($stmt)) {
    log_activity(
        $con,
        $userindex,
        "DB EXECUTE FAILED",
        $path,
        mysqli_stmt_error($stmt)
    );

    echo json_encode(["ok" => false, "error" => "Execute failed"]);
    mysqli_stmt_close($stmt);
    exit;
}

$result = mysqli_stmt_get_result($stmt);

if (!$result || mysqli_num_rows($result) === 0) {
    log_activity(
        $con,
        $userindex,
        "NO RECORD FOUND",
        $path,
        json_encode([
            "product_index" => $product_index,
            "sum_assured" => $sum_assured_str,
            "no" => $no
        ])
    );

    echo json_encode(["ok" => false, "error" => "No record found"]);
    mysqli_stmt_close($stmt);
    exit;
}

$row = mysqli_fetch_assoc($result);
mysqli_stmt_close($stmt);

/* =========================
   SUCCESS RESPONSE + LOG
========================= */
$response = [
    "ok" => true,
    "product_index" => $product_index,
    "product_name" => (string)$row["product_name"],
    "total_premium" => (float)$row["premium"]
];

log_activity(
    $con,
    $userindex,
    "PREMIUM FETCH SUCCESS",
    $path,
    json_encode($response)
);

echo json_encode($response);