<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header("Content-Type: application/json; charset=utf-8");

include("database.php"); // must create $con (mysqli)
date_default_timezone_set("Africa/Accra");

$path = $_SERVER['REQUEST_URI'] ?? 'scriptC_debit_customer.php';

/* =========================
   LOG FUNCTION
   Table: users_logs(userindex, caption, logs_date, path, data_inf)
   - userindex: mobil_no (customer identifier)
========================= */
function log_activity($con, $userindex, $caption, $path, $data_inf) {
    if (!isset($con) || !$con) return;

    $sql = "INSERT INTO users_logs (userindex, caption, logs_date, path, data_inf)
            VALUES (?, ?, NOW(), ?, ?)";

    if ($stmt = mysqli_prepare($con, $sql)) {
        $u = (string)$userindex;
        $c = (string)$caption;
        $p = (string)$path;
        $d = (string)$data_inf;

        mysqli_stmt_bind_param($stmt, "ssss", $u, $c, $p, $d);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
    }
}

function respond($con, $userindex, $path, $code, $payload, $caption = null) {
    http_response_code($code);

    if ($caption !== null) {
        log_activity($con, $userindex, $caption, $path, json_encode($payload));
    }

    echo json_encode($payload);
    exit;
}

/* =========================
   METHOD + DB CHECK
========================= */
if ($_SERVER["REQUEST_METHOD"] !== "POST") {
    if (isset($con) && $con) {
        log_activity($con, "UNKNOWN", "INVALID REQUEST METHOD", $path, json_encode([
            "method" => $_SERVER["REQUEST_METHOD"] ?? ""
        ]));
    }
    echo json_encode(["ok" => false, "message" => "Method not allowed"]);
    exit;
}

if (!isset($con) || !$con) {
    echo json_encode(["ok" => false, "message" => "Database connection failed"]);
    exit;
}

/* =========================
   READ JSON OR POST
========================= */
$raw  = file_get_contents("php://input");
$data = json_decode($raw, true);
$isJson = is_array($data);
$request = $isJson ? $data : $_POST;

// Inputs
$policy_number = trim((string)($request['policy_number'] ?? ""));
$mobil_no      = trim((string)($request['mobil_no'] ?? "")); // keep field name mobil_no
$amount_raw    = (string)($request['amount'] ?? "");

// ✅ normalize input momo_network, then apply TELECEL -> VODAFONE mapping
$momo_network  = strtoupper(trim((string)($request['momo_network'] ?? "")));
if ($momo_network === "TELECEL") {
    $momo_network = "VODAFONE";
}

$userindex = ($mobil_no !== "") ? $mobil_no : "UNKNOWN";

// Log request received (never log apiKey)
log_activity($con, $userindex, "REQUEST RECEIVED", $path, json_encode([
    "policy_number" => $policy_number,
    "mobil_no" => $mobil_no,
    "amount" => $amount_raw,
    "momo_network" => $momo_network
]));

/* =========================
   VALIDATION
========================= */
if ($policy_number === "" || $mobil_no === "" || $amount_raw === "" || $momo_network === "") {
    respond($con, $userindex, $path, 400, [
        "ok" => false,
        "message" => "Missing required fields",
        "required" => ["policy_number", "mobil_no", "amount", "momo_network"]
    ], "VALIDATION FAILED (missing fields)");
}

$amount = (float)$amount_raw;
if ($amount <= 0) {
    respond($con, $userindex, $path, 400, [
        "ok" => false,
        "message" => "Invalid amount"
    ], "VALIDATION FAILED (invalid amount)");
}

// ✅ since TELECEL is mapped to VODAFONE, allow list can exclude TELECEL if you want,
// but keeping it doesn't hurt. (Mapping happens before this check anyway.)
$allowed_networks = ["MTN", "AIRTEL", "TELECEL", "VODAFONE"];
if (!in_array($momo_network, $allowed_networks, true)) {
    respond($con, $userindex, $path, 400, [
        "ok" => false,
        "message" => "Invalid momo_network",
        "allowed" => $allowed_networks
    ], "VALIDATION FAILED (invalid network)");
}

/* =========================
   GENERATE REF (SAFE)
========================= */
mysqli_begin_transaction($con);

try {
    // lock row to avoid duplicates
    $sel = mysqli_prepare($con, "SELECT api_num FROM generate_idn WHERE generateindex = '1' LIMIT 1 FOR UPDATE");
    if (!$sel) throw new Exception("Prepare failed (select api_num): " . mysqli_error($con));

    if (!mysqli_stmt_execute($sel)) {
        $err = mysqli_stmt_error($sel);
        mysqli_stmt_close($sel);
        throw new Exception("Execute failed (select api_num): " . $err);
    }

    $res = mysqli_stmt_get_result($sel);
    mysqli_stmt_close($sel);

    if (!$res || mysqli_num_rows($res) === 0) {
        throw new Exception("generate_idn row not found for generateindex=1");
    }

    $row = mysqli_fetch_assoc($res);
    $api_num = (int)$row['api_num'];
    $api_num_next = $api_num + 1;

    $upd = mysqli_prepare($con, "UPDATE generate_idn SET api_num = ? WHERE generateindex = '1'");
    if (!$upd) throw new Exception("Prepare failed (update api_num): " . mysqli_error($con));

    mysqli_stmt_bind_param($upd, "i", $api_num_next);

    if (!mysqli_stmt_execute($upd)) {
        $err = mysqli_stmt_error($upd);
        mysqli_stmt_close($upd);
        throw new Exception("Execute failed (update api_num): " . $err);
    }

    mysqli_stmt_close($upd);

    // refNo = <api_num_next>-<policy_number>
    $refNo = $api_num_next . "-" . $policy_number;

    mysqli_commit($con);

    log_activity($con, $userindex, "REF GENERATED", $path, json_encode([
        "api_num_prev" => $api_num,
        "api_num_next" => $api_num_next,
        "refNo" => $refNo
    ]));

} catch (Exception $e) {
    mysqli_rollback($con);
    respond($con, $userindex, $path, 500, [
        "ok" => false,
        "message" => "Failed to generate reference number",
        "details" => $e->getMessage()
    ], "REF GENERATION FAILED");
}

/* =========================
   GATEWAY CONFIG
========================= */
$merchantId = '1664';
$productId  = '1';
$apiKey     = 'R09u4zJkYuNcZHGs6s3I7Dv5repVIf9x';

$narration  = "Debit {$momo_network} Customer";

// Voucher only for TELECEL/VODAFONE
// ✅ NOTE: since TELECEL is mapped to VODAFONE, TELECEL will also come here as VODAFONE.
$needsVoucher = in_array($momo_network, ["TELECEL", "VODAFONE"], true);
$voucher = $needsVoucher ? (string)($request['voucher'] ?? '874477') : null;

// Build payload
$payload = [
    "merchantId" => $merchantId,
    "productId"  => $productId,
    "msisdn"     => $mobil_no,
    "amount"     => (string)$amount,
    "network"    => $momo_network,
    "narration"  => $narration,
    "apiKey"     => $apiKey,
    "refNo"      => $refNo
];

if ($needsVoucher) {
    $payload["voucher"] = $voucher;
}

// Log outbound request (mask apiKey)
$payload_log = $payload;
$payload_log["apiKey"] = "***MASKED***";
log_activity($con, $userindex, "GATEWAY REQUEST PREPARED", $path, json_encode($payload_log));

/* =========================
   CURL CALL
========================= */
$curl = curl_init();

curl_setopt_array($curl, [
    CURLOPT_URL => "https://uniwallet.transflowitc.com/uniwallet/debit/customer",
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_ENCODING => "",
    CURLOPT_MAXREDIRS => 10,
    CURLOPT_TIMEOUT => 60,
    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
    CURLOPT_SSL_VERIFYHOST => 0,
    CURLOPT_SSL_VERIFYPEER => 0,
    CURLOPT_CUSTOMREQUEST => "POST",
    CURLOPT_POSTFIELDS => json_encode($payload),
    CURLOPT_HTTPHEADER => [
        "cache-control: no-cache",
        "content-type: application/json"
    ],
]);

$response = curl_exec($curl);
$err      = curl_error($curl);
$httpCode = (int)curl_getinfo($curl, CURLINFO_HTTP_CODE);

curl_close($curl);

if ($err) {
    respond($con, $userindex, $path, 500, [
        "ok" => false,
        "message" => "cURL error",
        "details" => $err,
        "refNo" => $refNo
    ], "CURL ERROR");
}

// Decode response if JSON
$decoded = json_decode($response, true);

if (json_last_error() !== JSON_ERROR_NONE) {
    log_activity($con, $userindex, "GATEWAY RESPONSE (NON-JSON)", $path, json_encode([
        "http_code" => $httpCode,
        "raw" => $response,
        "refNo" => $refNo
    ]));

    respond($con, $userindex, $path, 200, [
        "ok" => true,
        "refNo" => $refNo,
        "gateway_http_code" => $httpCode,
        "gateway_raw" => $response
    ], "SUCCESS (RAW GATEWAY RESPONSE)");
}

log_activity($con, $userindex, "GATEWAY RESPONSE (JSON)", $path, json_encode([
    "http_code" => $httpCode,
    "refNo" => $refNo,
    "gateway_response" => $decoded
]));

// Forward gateway response
respond($con, $userindex, $path, 200, [
    "ok" => true,
    "refNo" => $refNo,
    "gateway_http_code" => $httpCode,
    "gateway_response" => $decoded
], "SUCCESS (GATEWAY RESPONSE FORWARDED)");
?>