<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header("Content-Type: application/json; charset=utf-8");

require_once  "database.php";     // creates $con
require_once  "temp_polnum_function.php";
require_once  "mandate_creation.php";

date_default_timezone_set("Africa/Accra");
$sysdate = date("Y-m-d H:i:s");
$onboarding_channel = "USSD";
$path = $_SERVER['REQUEST_URI'] ?? 'scriptA_subscription.php';

function respond($code, $payload) {
    http_response_code($code);
    echo json_encode($payload);
    exit;
}

// --------- LOG FUNCTION (userindex = user mobile number) ----------
function log_activity($con, $user_mobile, $caption, $path, $data_inf) {
    if (!isset($con) || !$con) return;

    $sql = "INSERT INTO users_logs (userindex, caption, logs_date, path, data_inf)
            VALUES (?, ?, NOW(), ?, ?)";

    if ($stmt = mysqli_prepare($con, $sql)) {
        $u = (string)$user_mobile;
        $c = (string)$caption;
        $p = (string)$path;
        $d = (string)$data_inf;

        mysqli_stmt_bind_param($stmt, "ssss", $u, $c, $p, $d);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
    }
}

/**
 * ✅ Calls Script A (create_mandate) and returns JSON response
 */
function call_mandate_create(mysqli $con, int $userindex_int, string $momo_network, string $phoneNumber, string $temp_policy_num, string $frequencyType, string $amountToDebit, string $path) : array
{
    // create_mandate already logs internally using add_user_log()
    $result = create_mandate(
        $con,
        $userindex_int,
        $momo_network,
        $phoneNumber,
        $temp_policy_num,
        $frequencyType,
        $amountToDebit,
        $path
    );

    return $result;
}

// ---------- METHOD CHECK ----------
if ($_SERVER["REQUEST_METHOD"] !== "POST") {
    log_activity($con, "UNKNOWN", "INVALID REQUEST METHOD", $path, json_encode(["method" => $_SERVER["REQUEST_METHOD"] ?? ""]));
    respond(405, ["ok" => false, "error" => "Method not allowed. Use POST only."]);
}

if (!isset($con) || !$con) {
    respond(500, ["ok" => false, "error" => "Database connection not available. Check database.php"]);
}

// ---------- Read JSON or POST ----------
$raw  = file_get_contents("php://input");
$data = json_decode($raw, true);
$isJson = is_array($data);

$mobile_money_number = trim($isJson ? ($data["mobile_money_number"] ?? "") : ($_POST["mobile_money_number"] ?? ""));
$full_name           = trim($isJson ? ($data["full_name"] ?? "") : ($_POST["full_name"] ?? ""));
$date_of_birth       = trim($isJson ? ($data["date_of_birth"] ?? "") : ($_POST["date_of_birth"] ?? ""));
$gender              = trim($isJson ? ($data["gender"] ?? "") : ($_POST["gender"] ?? ""));

$sum_assured     = $isJson ? ($data["sum_assured"] ?? "") : ($_POST["sum_assured"] ?? "");
$product_id      = $isJson ? ($data["product_id"] ?? "") : ($_POST["product_id"] ?? "");
$total_premium   = $isJson ? ($data["total_premium"] ?? "") : ($_POST["total_premium"] ?? "");
$momo_network    = trim($isJson ? ($data["momo_network"] ?? "") : ($_POST["momo_network"] ?? ""));
$product_name    = $isJson ? ($data["product_name"] ?? "") : ($_POST["product_name"] ?? "");

// optional inputs for mandate
$frequencyType   = trim($isJson ? ($data["frequencyType"] ?? "") : ($_POST["frequencyType"] ?? ""));
if ($frequencyType === "") $frequencyType = "MONTHLY";

// Additional lives
$add_momo   = $isJson ? ($data["addlife_mobile_money_number"] ?? []) : ($_POST["addlife_mobile_money_number"] ?? []);
$add_name   = $isJson ? ($data["addlife_full_name"] ?? [])          : ($_POST["addlife_full_name"] ?? []);
$add_dob    = $isJson ? ($data["addlife_date_of_birth"] ?? [])      : ($_POST["addlife_date_of_birth"] ?? []);
$add_gender = $isJson ? ($data["addlife_gender"] ?? [])             : ($_POST["addlife_gender"] ?? []);
$add_rel    = $isJson ? ($data["addlife_relationship"] ?? [])       : ($_POST["addlife_relationship"] ?? []);

if (!is_array($add_momo))   $add_momo = [];
if (!is_array($add_name))   $add_name = [];
if (!is_array($add_dob))    $add_dob = [];
if (!is_array($add_gender)) $add_gender = [];
if (!is_array($add_rel))    $add_rel = [];

// Log request received
log_activity($con, ($mobile_money_number !== "" ? $mobile_money_number : "UNKNOWN"), "REQUEST RECEIVED", $path, json_encode([
    "mobile_money_number" => $mobile_money_number,
    "full_name" => $full_name,
    "date_of_birth" => $date_of_birth,
    "gender" => $gender,
    "sum_assured" => $sum_assured,
    "product_id" => $product_id,
    "total_premium" => $total_premium,
    "momo_network" => $momo_network,
    "product_name" => $product_name,
    "frequencyType" => $frequencyType,
    "additional_lives_count" => is_array($add_momo) ? count($add_momo) : 0
]));

// Normalize types
$product_id       = (int)$product_id;
$sum_assured_f    = (float)$sum_assured;
$total_premium_f  = (float)$total_premium;

// ✅ userindex for create_mandate must be int.
// If you have a real session userindex, use it. Otherwise keep 0.
$userindex_int = 0;

// ---------- Validate required (for NEW insert only) ----------
$has_minimum_for_insert = !(
    $mobile_money_number === "" || $full_name === "" || $date_of_birth === "" || $gender === "" ||
    $product_id <= 0 || $sum_assured_f <= 0 || $total_premium_f <= 0 || $momo_network === ""
);

// =====================================================
// ✅ 1) EXISTS CHECK
// If exists: DO NOT ERROR, go straight to mandate creation
// =====================================================
$chk = mysqli_prepare($con, "SELECT subscriptions_id, policy_number, total_premium, momo_network FROM subscription_main WHERE mobile_money_number = ? LIMIT 1");
if (!$chk) {
    log_activity($con, $mobile_money_number ?: "UNKNOWN", "DB PREPARE FAILED (exists check)", $path, mysqli_error($con));
    respond(500, ["ok"=>false,"error"=>"Prepare failed (exists check)","details"=>mysqli_error($con)]);
}

mysqli_stmt_bind_param($chk, "s", $mobile_money_number);
mysqli_stmt_execute($chk);
$chkRes = mysqli_stmt_get_result($chk);

if ($chkRes && mysqli_num_rows($chkRes) > 0) {
    $existing = mysqli_fetch_assoc($chkRes);
    mysqli_stmt_close($chk);

    $subscriptions_id_existing = (int)($existing["subscriptions_id"] ?? 0);
    $policy_number_existing    = (string)($existing["policy_number"] ?? "");
    $db_total_premium          = (string)($existing["total_premium"] ?? "");
    $db_momo_network           = (string)($existing["momo_network"] ?? $momo_network);

    // amountToDebit priority: request total_premium > db total_premium
    $amountToDebit = ((float)$total_premium_f > 0) ? (string)$total_premium_f : (string)$db_total_premium;

    log_activity($con, $mobile_money_number, "SUBSCRIPTION EXISTS -> CALLING MANDATE", $path, json_encode([
        "subscriptions_id" => $subscriptions_id_existing,
        "policy_number" => $policy_number_existing,
        "amountToDebit" => $amountToDebit,
        "momo_network" => $db_momo_network,
        "frequencyType" => $frequencyType
    ]));

    // ✅ Call Script A now
    $mandate = call_mandate_create(
        $con,
        $userindex_int,
        $db_momo_network,
        $mobile_money_number,
        $policy_number_existing,
        $frequencyType,
        $amountToDebit,
        $path
    );

    if (!($mandate["ok"] ?? false)) {
        respond(200, [
            "ok" => false,
            "status" => "EXISTS_BUT_MANDATE_FAILED",
            "subscriptions_id" => $subscriptions_id_existing,
            "policy_number" => $policy_number_existing,
            "mandate" => $mandate
        ]);
    }

    respond(200, [
        "ok" => true,
        "status" => "EXISTS_MANDATE_CREATED",
        "subscriptions_id" => $subscriptions_id_existing,
        "policy_number" => $policy_number_existing,
        "mandate" => $mandate
    ]);
}
mysqli_stmt_close($chk);

// =====================================================
// ✅ If NOT exists, then we must have required fields to insert
// =====================================================
if (!$has_minimum_for_insert) {
    log_activity($con, ($mobile_money_number !== "" ? $mobile_money_number : "UNKNOWN"), "VALIDATION FAILED (new insert)", $path, json_encode([
        "missing_or_invalid" => true,
        "mobile_money_number" => $mobile_money_number,
        "product_id" => $product_id,
        "sum_assured" => $sum_assured,
        "total_premium" => $total_premium,
        "momo_network" => $momo_network
    ]));

    respond(400, [
        "ok" => false,
        "error" => "Missing required fields for new subscription",
        "required" => ["mobile_money_number","full_name","date_of_birth","gender","sum_assured","product_id","total_premium","momo_network"]
    ]);
}

mysqli_begin_transaction($con);

// 2) Get premium
$premStmt = mysqli_prepare($con, "SELECT premium_main_life FROM cover_levels WHERE product_index = ? AND sum_assured = ? LIMIT 1");
if (!$premStmt) {
    log_activity($con, $mobile_money_number, "DB PREPARE FAILED (get premium)", $path, mysqli_error($con));
    mysqli_rollback($con);
    respond(500, ["ok"=>false,"error"=>"Prepare failed (get premium)","details"=>mysqli_error($con)]);
}

mysqli_stmt_bind_param($premStmt, "id", $product_id, $sum_assured_f);
mysqli_stmt_execute($premStmt);
$premRes = mysqli_stmt_get_result($premStmt);

if (!$premRes || mysqli_num_rows($premRes) === 0) {
    log_activity($con, $mobile_money_number, "PREMIUM NOT FOUND", $path, json_encode([
        "product_id" => $product_id,
        "sum_assured" => $sum_assured_f
    ]));
    mysqli_stmt_close($premStmt);
    mysqli_rollback($con);
    respond(404, ["ok"=>false,"error"=>"Premium not found","inputs"=>["product_id"=>$product_id,"sum_assured"=>$sum_assured_f]]);
}
$premiumRow = mysqli_fetch_assoc($premRes);
$premium = (float)$premiumRow["premium_main_life"];
mysqli_stmt_close($premStmt);

log_activity($con, $mobile_money_number, "PREMIUM FETCHED", $path, json_encode([
    "product_id" => $product_id,
    "sum_assured" => $sum_assured_f,
    "premium" => $premium
]));

// =====================================================
// ✅ 2.5) GENERATE POLICY NUMBER + UPDATE generate_idn
// =====================================================
$temp_policy_num = generateTempPolicyNumber($con);

$policy_number    = $temp_policy_num;
$cc_policy_number = $temp_policy_num;

log_activity($con, $mobile_money_number, "POLICY NUMBER GENERATED", $path, json_encode([
    "policy_number" => $policy_number,
    "cc_policy_number" => $cc_policy_number
]));

$stmtGen = $con->prepare("
    UPDATE generate_idn
    SET temp_policy_num = ?
    WHERE generateindex = 1
");
if (!$stmtGen) {
    log_activity($con, $mobile_money_number, "DB PREPARE FAILED (update temp_policy_num)", $path, $con->error);
    mysqli_rollback($con);
    respond(500, ["ok"=>false,"error"=>"Prepare failed (update temp_policy_num)","details"=>$con->error]);
}
$stmtGen->bind_param("s", $policy_number);
if (!$stmtGen->execute()) {
    $err = $stmtGen->error;
    $stmtGen->close();
    log_activity($con, $mobile_money_number, "UPDATE FAILED (temp_policy_num)", $path, $err);
    mysqli_rollback($con);
    respond(500, ["ok"=>false,"error"=>"Failed to update temp_policy_num","details"=>$err]);
}
$stmtGen->close();

// 3) Insert subscription_main
$sqlMain = "
INSERT INTO subscription_main
(policy_number, cc_policy_number, mobile_money_number, full_name, date_of_birth, gender, product_name, premium, total_premium, momo_network, onboarding_channel, sysdate)
VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
";

$insMain = mysqli_prepare($con, $sqlMain);
if (!$insMain) {
    log_activity($con, $mobile_money_number, "DB PREPARE FAILED (insert subscription_main)", $path, mysqli_error($con));
    mysqli_rollback($con);
    respond(500, [
        "ok" => false,
        "error" => "Prepare failed (insert subscription_main)",
        "details" => mysqli_error($con),
        "hint" => "Confirm subscription_main has columns: policy_number, cc_policy_number."
    ]);
}

mysqli_stmt_bind_param(
    $insMain,
    "sssssssddsss",
    $policy_number,
    $cc_policy_number,
    $mobile_money_number,
    $full_name,
    $date_of_birth,
    $gender,
    $product_name,
    $premium,
    $total_premium_f,
    $momo_network,
    $onboarding_channel,
    $sysdate
);

if (!mysqli_stmt_execute($insMain)) {
    $err = mysqli_stmt_error($insMain);
    log_activity($con, $mobile_money_number, "INSERT FAILED (subscription_main)", $path, $err);
    mysqli_stmt_close($insMain);
    mysqli_rollback($con);
    respond(500, ["ok"=>false,"error"=>"Failed to insert subscription_main","details"=>$err]);
}

mysqli_stmt_close($insMain);
$subscriptions_id = mysqli_insert_id($con);

log_activity($con, $mobile_money_number, "SUBSCRIPTION MAIN INSERTED", $path, json_encode([
    "subscriptions_id" => $subscriptions_id,
    "policy_number" => $policy_number,
    "cc_policy_number" => $cc_policy_number
]));

// 4) Insert additional lives (optional)
$inserted_additional = 0;

if (count($add_momo) > 0) {
    $insAdd = mysqli_prepare($con, "
        INSERT INTO subscription_additionallife
        (subscriptions_id, mobile_money_number, full_name, date_of_birth, gender, relationship, date)
        VALUES (?, ?, ?, ?, ?, ?, ?)
    ");
    if (!$insAdd) {
        log_activity($con, $mobile_money_number, "DB PREPARE FAILED (insert additional lives)", $path, mysqli_error($con));
        mysqli_rollback($con);
        respond(500, ["ok"=>false,"error"=>"Prepare failed (insert additional lives)","details"=>mysqli_error($con)]);
    }

    for ($i=0; $i<count($add_momo); $i++) {
        $m = trim($add_momo[$i] ?? "");
        $n = trim($add_name[$i] ?? "");
        $d = trim($add_dob[$i] ?? "");
        $g = trim($add_gender[$i] ?? "");
        $r = trim($add_rel[$i] ?? "");

        if ($m==="" && $n==="" && $d==="" && $g==="" && $r==="") continue;

        mysqli_stmt_bind_param($insAdd, "issssss", $subscriptions_id, $m, $n, $d, $g, $r, $sysdate);

        if (!mysqli_stmt_execute($insAdd)) {
            $err = mysqli_stmt_error($insAdd);
            log_activity($con, $mobile_money_number, "INSERT FAILED (additional life)", $path, json_encode([
                "row_index" => $i,
                "details" => $err,
                "row_data" => ["mobile_money_number"=>$m,"full_name"=>$n,"dob"=>$d,"gender"=>$g,"relationship"=>$r]
            ]));
            mysqli_stmt_close($insAdd);
            mysqli_rollback($con);
            respond(500, ["ok"=>false,"error"=>"Failed to insert additional life","row_index"=>$i,"details"=>$err]);
        }

        $inserted_additional++;
    }

    mysqli_stmt_close($insAdd);

    log_activity($con, $mobile_money_number, "ADDITIONAL LIVES INSERTED", $path, json_encode([
        "additional_lives_inserted" => $inserted_additional
    ]));
}

mysqli_commit($con);

log_activity($con, $mobile_money_number, "SUBSCRIPTION CREATED SUCCESSFULLY", $path, json_encode([
    "subscriptions_id" => $subscriptions_id,
    "policy_number" => $policy_number,
    "cc_policy_number" => $cc_policy_number,
    "premium" => $premium,
    "additional_lives_inserted" => $inserted_additional
]));

// =====================================================
// ✅ 5) AFTER SUCCESSFUL SAVE -> CALL SCRIPT A (MANDATE)
// =====================================================
$amountToDebit = (string)$total_premium_f;

log_activity($con, $mobile_money_number, "CALLING MANDATE AFTER SAVE", $path, json_encode([
    "policy_number" => $policy_number,
    "amountToDebit" => $amountToDebit,
    "momo_network" => $momo_network,
    "frequencyType" => $frequencyType
]));

$mandate = call_mandate_create(
    $con,
    $userindex_int,
    $momo_network,
    $mobile_money_number,
    $policy_number,
    $frequencyType,
    $amountToDebit,
    $path
);

if (!($mandate["ok"] ?? false)) {
    respond(200, [
        "ok" => false,
        "status" => "SAVED_BUT_MANDATE_FAILED",
        "message" => "Subscription saved successfully but mandate creation failed",
        "subscriptions_id" => $subscriptions_id,
        "policy_number" => $policy_number,
        "cc_policy_number" => $cc_policy_number,
        "premium" => $premium,
        "additional_lives_inserted" => $inserted_additional,
        "mandate" => $mandate
    ]);
}

respond(200, [
    "ok" => true,
    "status" => "SAVED_AND_MANDATE_CREATED",
    "message" => "Subscription created successfully and mandate created",
    "subscriptions_id" => $subscriptions_id,
    "policy_number" => $policy_number,
    "cc_policy_number" => $cc_policy_number,
    "premium" => $premium,
    "additional_lives_inserted" => $inserted_additional,
    "mandate" => $mandate
]);
?>