<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header("Content-Type: application/json; charset=utf-8");

include("database.php"); // must create $con (mysqli)
date_default_timezone_set("Africa/Accra");

$path = $_SERVER['REQUEST_URI'] ?? 'sum_assured_by_product.php';

/* =========================
   LOG FUNCTION
   Table: users_logs(userindex, caption, logs_date, path, data_inf)
   - userindex: pass mobile_money_number OR userindex; if none, "UNKNOWN"
========================= */
function log_activity($con, $userindex, $caption, $path, $data_inf) {
    if (!isset($con) || !$con) return;

    $sql = "INSERT INTO users_logs (userindex, caption, logs_date, path, data_inf)
            VALUES (?, ?, NOW(), ?, ?)";

    if ($stmt = mysqli_prepare($con, $sql)) {
        $u = (string)$userindex;
        $c = (string)$caption;
        $p = (string)$path;
        $d = (string)$data_inf;

        mysqli_stmt_bind_param($stmt, "ssss", $u, $c, $p, $d);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
    }
}

function respond($con, $userindex, $path, $code, $payload, $caption = null) {
    http_response_code($code);
    if ($caption !== null) {
        log_activity($con, $userindex, $caption, $path, json_encode($payload));
    }
    echo json_encode($payload);
    exit;
}

/* =========================
   METHOD + DB CHECK
========================= */
if ($_SERVER["REQUEST_METHOD"] !== "GET" && $_SERVER["REQUEST_METHOD"] !== "POST") {
    if (isset($con) && $con) {
        log_activity($con, "UNKNOWN", "INVALID REQUEST METHOD", $path, json_encode([
            "method" => $_SERVER["REQUEST_METHOD"] ?? ""
        ]));
    }
    echo json_encode(["ok" => false, "message" => "Method not allowed"]);
    exit;
}

if (!isset($con) || !$con) {
    echo json_encode(["ok" => false, "message" => "Database connection failed"]);
    exit;
}

/* =========================
   READ INPUT (JSON or REQUEST)
========================= */
$raw  = file_get_contents("php://input");
$data = json_decode($raw, true);
$isJson = is_array($data);

$product_index = $isJson ? ($data["product_index"] ?? "") : ($_REQUEST["product_index"] ?? "");

// optional identifier for logging
$userindex = trim(
    $isJson
        ? ($data["userindex"] ?? ($data["mobile_money_number"] ?? ""))
        : ($_REQUEST["userindex"] ?? ($_REQUEST["mobile_money_number"] ?? ""))
);
if ($userindex === "") $userindex = "UNKNOWN";

log_activity($con, $userindex, "REQUEST RECEIVED", $path, json_encode([
    "product_index" => $product_index,
    "method" => $_SERVER["REQUEST_METHOD"] ?? ""
]));

/* =========================
   VALIDATION
========================= */
$product_index = (int)$product_index;
if ($product_index <= 0) {
    respond($con, $userindex, $path, 400, [
        "ok" => false,
        "message" => "Invalid product_index"
    ], "VALIDATION FAILED (invalid product_index)");
}

/* =========================
   QUERY SUM ASSURED
========================= */
$sql = "
    SELECT DISTINCT sum_assured
    FROM cover_levels
    WHERE product_index = ?
    ORDER BY sum_assured ASC
";

$stmt = mysqli_prepare($con, $sql);
if (!$stmt) {
    respond($con, $userindex, $path, 500, [
        "ok" => false,
        "message" => "Prepare failed",
        "details" => mysqli_error($con)
    ], "DB PREPARE FAILED");
}

mysqli_stmt_bind_param($stmt, "i", $product_index);

if (!mysqli_stmt_execute($stmt)) {
    $err = mysqli_stmt_error($stmt);
    mysqli_stmt_close($stmt);
    respond($con, $userindex, $path, 500, [
        "ok" => false,
        "message" => "Execute failed",
        "details" => $err
    ], "DB EXECUTE FAILED");
}

$res = mysqli_stmt_get_result($stmt);
mysqli_stmt_close($stmt);

$sum_assured_list = [];
if ($res) {
    while ($row = mysqli_fetch_assoc($res)) {
        $sum_assured_list[] = (float)$row["sum_assured"];
    }
}

if (count($sum_assured_list) === 0) {
    respond($con, $userindex, $path, 404, [
        "ok" => false,
        "message" => "No sum_assured found for this product_index",
        "product_index" => $product_index
    ], "NO RECORD FOUND");
}

/* =========================
   SUCCESS
========================= */
$response = [
    "ok" => true,
    "product_index" => $product_index,
    "count" => count($sum_assured_list),
    "sum_assured" => $sum_assured_list
];

log_activity($con, $userindex, "SUM ASSURED FETCH SUCCESS", $path, json_encode([
    "product_index" => $product_index,
    "count" => $response["count"]
]));

echo json_encode($response);
?>