
<?php
/**
 * Generate Monthly Reset Policy Number using table: generate_id, field: temp_policy_num
 * Format: MMYY0000000001  (e.g. 01260000000001)
 * Resets sequence when month changes (01 -> 02 etc.)
 *
 * Table: generate_id
 * Field used: temp_policy_num
 *
 * @param mysqli $con
 * @return string
 * @throws Exception
 */
function generateTempPolicyNumber(mysqli $con): string
{
    date_default_timezone_set('Africa/Accra');

    // Prefix MMYY e.g. 0126
    $prefix = date('m') . date('y');

    // Get the latest temp_policy_num for this month/year prefix
    $stmt = $con->prepare("
        SELECT temp_policy_num
        FROM generate_idn
        WHERE temp_policy_num LIKE CONCAT(?, '%')
        ORDER BY temp_policy_num DESC
        LIMIT 1
    ");

    if (!$stmt) {
        throw new Exception("Prepare failed: " . $con->error);
    }

    $stmt->bind_param("s", $prefix);
    $stmt->execute();
    $stmt->bind_result($lastTempPolicyNum);
    $stmt->fetch();
    $stmt->close();

    // Determine next sequence
    if (!empty($lastTempPolicyNum)) {
        // last 10 digits are the sequence
        $lastSeq = (int) substr($lastTempPolicyNum, -10);
        $nextSeq = $lastSeq + 1;
    } else {
        // first number in the month
        $nextSeq = 1;
    }

    // Pad to 10 digits
    $seq = str_pad((string)$nextSeq, 10, '0', STR_PAD_LEFT);

    // Final policy number
    return $prefix . $seq;
}

/**
 * OPTIONAL helper: create customer_id in your sample style: ABR + temp_policy_num
 * Example: temp_policy_num=01260000000003 => customer_id=ABR01260000003
 */
function buildCustomerIdFromTemp(string $temp_policy_num): string
{
    return "ABR" . $temp_policy_num;
}
?>