<?php
/**
 * Abrabo Pa Insurance USSD Application
 * USSD Code: *712*100#
 * 
 * This application handles the registration flow only
 * Other menu options will be implemented soon
 */

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Start session for storing USSD data
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Database configuration
$db_config = [
    'host' => 'localhost',
    'user' => 'root',
    'pass' => '',
    'name' => 'abrabo_insurance',
    'port' => 3306
];

// API base URL
define('API_BASE_URL', 'http://localhost/abrabo_pa_ussd/api/');

// Create database connection
$conn = createDatabaseConnection($db_config);

// Create necessary tables
createTables($conn);

// Get USSD parameters
$session_id = $_GET['session_id'] ?? uniqid('ussd_', true);
$msisdn = $_GET['msisdn'] ?? '';
$network = $_GET['network'] ?? 'MTN';
$ussd_code = $_GET['ussd_code'] ?? '*712*100#';
$user_input = $_GET['user_input'] ?? '';

// Clean and validate phone number
$msisdn = cleanPhoneNumber($msisdn);

// Initialize session if not exists
if (!isset($_SESSION['ussd_session'])) {
    $_SESSION['ussd_session'] = [
        'session_id' => $session_id,
        'msisdn' => $msisdn,
        'network' => $network,
        'ussd_code' => $ussd_code,
        'menu_level' => 'main',
        'step' => 0,
        'data' => [],
        'created_at' => date('Y-m-d H:i:s'),
        'last_activity' => time()
    ];
    
    // Log session start
    logSession($conn, $session_id, $msisdn, $network, 'SESSION_START', 'Main menu displayed');
}

// Check session timeout (5 minutes)
if (time() - $_SESSION['ussd_session']['last_activity'] > 300) {
    session_destroy();
    echo "END Session expired. Please dial *712*100# again.";
    exit();
}

// Update last activity
$_SESSION['ussd_session']['last_activity'] = time();

// Handle USSD request
handleUssdRequest($conn, $user_input);

// Close database connection
$conn->close();

// ============================================================================
// MAIN REQUEST HANDLER
// ============================================================================

function handleUssdRequest($conn, $user_input) {
    $session = $_SESSION['ussd_session'];
    $menu_level = $session['menu_level'];
    $step = $session['step'];
    
    // Log user input
    if (!empty($user_input)) {
        logSession($conn, $session['session_id'], $session['msisdn'], $session['network'], 'USER_INPUT', $user_input);
    }
    
    // Route to appropriate handler
    switch ($menu_level) {
        case 'main':
            handleMainMenu($user_input);
            break;
            
        case 'register':
            handleRegistration($conn, $user_input, $step);
            break;
            
        case 'register_coming_soon':
            handleComingSoon($user_input);
            break;
            
        default:
            showError("Invalid session state. Please start again.");
    }
}

// ============================================================================
// MAIN MENU HANDLER
// ============================================================================

function handleMainMenu($user_input) {
    if (empty($user_input)) {
        // Initial request - show main menu
        $response = "CON Welcome to Abrabo Pa Insurance\n";
        $response .= "1. Register\n";
        $response .= "2. Pay Premium\n";
        $response .= "3. Stop Auto Deduct\n";
        $response .= "4. Claims\n";
        $response .= "5. Check Balance\n";
        $response .= "6. Terms and Conditions\n";
        $response .= "7. Helpline\n";
        
        echo $response;
    } else {
        // Handle menu selection
        switch ($user_input) {
            case '1':
                // Start registration process
                $_SESSION['ussd_session']['menu_level'] = 'register';
                $_SESSION['ussd_session']['step'] = 1;
                $_SESSION['ussd_session']['data'] = [
                    'main_life' => [],
                    'dependents' => [],
                    'registration_step' => 1
                ];
                
                echo "CON Enter your full name:";
                break;
                
            case '2':
            case '3':
            case '4':
            case '5':
            case '6':
            case '7':
                // Other menu items - coming soon
                $_SESSION['ussd_session']['menu_level'] = 'register_coming_soon';
                $_SESSION['ussd_session']['selected_option'] = $user_input;
                
                $option_names = [
                    '2' => 'Pay Premium',
                    '3' => 'Stop Auto Deduct',
                    '4' => 'Claims',
                    '5' => 'Check Balance',
                    '6' => 'Terms and Conditions',
                    '7' => 'Helpline'
                ];
                
                echo "END " . $option_names[$user_input] . " feature is coming soon!\n";
                echo "Currently only registration is available.";
                break;
                
            default:
                echo "END Invalid selection. Please dial *712*100# again.";
        }
    }
}

// ============================================================================
// REGISTRATION HANDLER
// ============================================================================

function handleRegistration($conn, $user_input, $step) {
    $data = $_SESSION['ussd_session']['data'];
    $registration_step = $data['registration_step'] ?? 1;
    
    switch ($registration_step) {
        case 1:
            // Step 1: Get full name
            if (empty(trim($user_input))) {
                echo "CON Full name cannot be empty. Please enter your full name:";
                return;
            }
            
            $data['main_life']['full_name'] = trim($user_input);
            $data['registration_step'] = 2;
            $_SESSION['ussd_session']['data'] = $data;
            
            echo "CON Enter your date of birth (YYYY-MM-DD):\nExample: 1990-01-15";
            break;
            
        case 2:
            // Step 2: Validate date of birth
            if (!validateDate($user_input)) {
                echo "CON Invalid date format. Please enter in YYYY-MM-DD format:\nExample: 1990-01-15";
                return;
            }
            
            // Call age validation API
            $api_response = callApi('age.php', [
                'life_type' => 'Main Life',
                'date_of_birth' => $user_input
            ]);
            
            if ($api_response && isset($api_response['ok']) && $api_response['ok'] === true) {
                $data['main_life']['date_of_birth'] = $user_input;
                $data['registration_step'] = 3;
                $_SESSION['ussd_session']['data'] = $data;
                
                echo "CON Select gender:\n";
                echo "1. Male\n";
                echo "2. Female";
            } else {
                echo "CON Sorry, age validation failed.\n";
                echo "You must be between 18-65 years old.\n";
                echo "Please enter a valid date of birth:";
            }
            break;
            
        case 3:
            // Step 3: Select gender
            if ($user_input === '1') {
                $data['main_life']['gender'] = 'Male';
            } elseif ($user_input === '2') {
                $data['main_life']['gender'] = 'Female';
            } else {
                echo "CON Invalid selection. Please select:\n";
                echo "1. Male\n";
                echo "2. Female";
                return;
            }
            
            $data['registration_step'] = 4;
            $_SESSION['ussd_session']['data'] = $data;
            
            echo "CON Do you want to add dependents?\n";
            echo "1. Yes\n";
            echo "2. No";
            break;
            
        case 4:
            // Step 4: Handle dependents choice
            if ($user_input === '1') {
                $data['has_dependents'] = true;
                $data['registration_step'] = 5;
                $_SESSION['ussd_session']['data'] = $data;
                
                echo "CON Enter number of dependents (1 or 2):";
            } elseif ($user_input === '2') {
                $data['has_dependents'] = false;
                $data['registration_step'] = 7; // Skip to product selection
                $_SESSION['ussd_session']['data'] = $data;
                
                // Get products from API
                $api_response = callApi('product.php', []);
                
                if ($api_response && isset($api_response['ok']) && $api_response['ok'] === true) {
                    $products = $api_response['products'] ?? [];
                    $data['available_products'] = $products;
                    $_SESSION['ussd_session']['data'] = $data;
                    
                    $response = "CON Select a product:\n";
                    $counter = 1;
                    foreach ($products as $product) {
                        $response .= $counter . ". " . $product['product_name'] . "\n";
                        $counter++;
                    }
                    echo $response;
                } else {
                    showError("Unable to load products. Please try again later.");
                }
            } else {
                echo "CON Invalid selection. Please select:\n";
                echo "1. Yes\n";
                echo "2. No";
            }
            break;
            
        case 5:
            // Step 5: Get number of dependents
            if (!in_array($user_input, ['1', '2'])) {
                echo "CON You can only add 1 or 2 dependents. Please enter 1 or 2:";
                return;
            }
            
            $data['num_dependents'] = (int)$user_input;
            $data['current_dependent'] = 1;
            $data['registration_step'] = 6;
            $_SESSION['ussd_session']['data'] = $data;
            
            echo "CON Enter Dependent 1 full name:";
            break;
            
        case 6:
            // Step 6: Collect dependent information
            handleDependentInfo($conn, $user_input, $data);
            break;
            
        case 7:
            // Step 7: Select product
            handleProductSelection($conn, $user_input, $data);
            break;
            
        case 8:
            // Step 8: Select sum assured
            handleSumAssuredSelection($conn, $user_input, $data);
            break;
            
        case 9:
            // Step 9: Calculate and show premium
            handlePremiumCalculation($conn, $user_input, $data);
            break;
            
        case 10:
            // Step 10: Final confirmation and registration
            handleFinalRegistration($conn, $user_input, $data);
            break;
            
        default:
            showError("Invalid registration step. Please start again.");
    }
}

// ============================================================================
// DEPENDENT INFORMATION HANDLER
// ============================================================================

function handleDependentInfo($conn, $user_input, &$data) {
    $current_dep = $data['current_dependent'] ?? 1;
    $max_deps = $data['num_dependents'] ?? 0;
    
    if (!isset($data['dependents'][$current_dep - 1])) {
        $data['dependents'][$current_dep - 1] = [];
    }
    
    $dep_data = &$data['dependents'][$current_dep - 1];
    
    // Determine which field we're collecting
    if (!isset($dep_data['full_name'])) {
        // Get dependent full name
        $dep_data['full_name'] = trim($user_input);
        $_SESSION['ussd_session']['data'] = $data;
        
        echo "CON Enter Dependent " . $current_dep . " date of birth (YYYY-MM-DD):";
        
    } elseif (!isset($dep_data['date_of_birth'])) {
        // Validate dependent DOB
        if (!validateDate($user_input)) {
            echo "CON Invalid date format. Please enter in YYYY-MM-DD format:\nExample: 2000-01-15";
            return;
        }
        
        // Validate dependent age
        $api_response = callApi('age.php', [
            'life_type' => 'Dependent',
            'date_of_birth' => $user_input
        ]);
        
        if ($api_response && isset($api_response['ok']) && $api_response['ok'] === true) {
            $dep_data['date_of_birth'] = $user_input;
            $_SESSION['ussd_session']['data'] = $data;
            
            echo "CON Select Dependent " . $current_dep . " gender:\n";
            echo "1. Male\n";
            echo "2. Female";
        } else {
            echo "CON Dependent age is not eligible. Please enter a different date of birth:";
        }
        
    } elseif (!isset($dep_data['gender'])) {
        // Get dependent gender
        if ($user_input === '1') {
            $dep_data['gender'] = 'Male';
        } elseif ($user_input === '2') {
            $dep_data['gender'] = 'Female';
        } else {
            echo "CON Invalid selection. Please select:\n";
            echo "1. Male\n";
            echo "2. Female";
            return;
        }
        
        $_SESSION['ussd_session']['data'] = $data;
        
        // Check if we have more dependents to add
        if ($current_dep < $max_deps) {
            $data['current_dependent'] = $current_dep + 1;
            $_SESSION['ussd_session']['data'] = $data;
            
            echo "CON Enter Dependent " . ($current_dep + 1) . " full name:";
        } else {
            // All dependents added, move to product selection
            $data['registration_step'] = 7;
            $_SESSION['ussd_session']['data'] = $data;
            
            // Get products from API
            $api_response = callApi('product.php', []);
            
            if ($api_response && isset($api_response['ok']) && $api_response['ok'] === true) {
                $products = $api_response['products'] ?? [];
                $data['available_products'] = $products;
                $_SESSION['ussd_session']['data'] = $data;
                
                $response = "CON Select a product:\n";
                $counter = 1;
                foreach ($products as $product) {
                    $response .= $counter . ". " . $product['product_name'] . "\n";
                    $counter++;
                }
                echo $response;
            } else {
                showError("Unable to load products. Please try again later.");
            }
        }
    }
}

// ============================================================================
// PRODUCT SELECTION HANDLER
// ============================================================================

function handleProductSelection($conn, $user_input, &$data) {
    $products = $data['available_products'] ?? [];
    $selected_index = (int)$user_input - 1;
    
    if ($selected_index < 0 || $selected_index >= count($products)) {
        $response = "CON Invalid selection. Please select:\n";
        $counter = 1;
        foreach ($products as $product) {
            $response .= $counter . ". " . $product['product_name'] . "\n";
            $counter++;
        }
        echo $response;
        return;
    }
    
    $selected_product = $products[$selected_index];
    $data['selected_product'] = $selected_product;
    $data['registration_step'] = 8;
    $_SESSION['ussd_session']['data'] = $data;
    
    // Get sum assured options for this product
    $api_response = callApi('sum_assured.php', [
        'product_index' => $selected_product['product_index']
    ]);
    
    if ($api_response && isset($api_response['ok']) && $api_response['ok'] === true) {
        $sum_assured_options = $api_response['sum_assured'] ?? [];
        $data['sum_assured_options'] = $sum_assured_options;
        $_SESSION['ussd_session']['data'] = $data;
        
        $response = "CON Select Sum Assured (GHS):\n";
        $counter = 1;
        foreach ($sum_assured_options as $amount) {
            $response .= $counter . ". GHS " . number_format($amount, 2) . "\n";
            $counter++;
        }
        echo $response;
    } else {
        showError("Unable to load sum assured options. Please try again.");
    }
}

// ============================================================================
// SUM ASSURED SELECTION HANDLER
// ============================================================================

function handleSumAssuredSelection($conn, $user_input, &$data) {
    $sum_assured_options = $data['sum_assured_options'] ?? [];
    $selected_index = (int)$user_input - 1;
    
    if ($selected_index < 0 || $selected_index >= count($sum_assured_options)) {
        $response = "CON Invalid selection. Please select:\n";
        $counter = 1;
        foreach ($sum_assured_options as $amount) {
            $response .= $counter . ". GHS " . number_format($amount, 2) . "\n";
            $counter++;
        }
        echo $response;
        return;
    }
    
    $selected_sum_assured = $sum_assured_options[$selected_index];
    $data['selected_sum_assured'] = $selected_sum_assured;
    $data['registration_step'] = 9;
    $_SESSION['ussd_session']['data'] = $data;
    
    // Calculate premium
    $selected_product = $data['selected_product'];
    $num_dependents = isset($data['has_dependents']) && $data['has_dependents'] ? 
                      ($data['num_dependents'] ?? 0) : 0;
    
    $api_response = callApi('get_premium.php', [
        'product_index' => $selected_product['product_index'],
        'sum_assured' => $selected_sum_assured,
        'no' => $num_dependents
    ]);
    
    if ($api_response && isset($api_response['ok']) && $api_response['ok'] === true) {
        $premium = $api_response['total_premium'] ?? 0;
        $data['premium'] = $premium;
        $data['registration_step'] = 10;
        $_SESSION['ussd_session']['data'] = $data;
        
        $response = "CON Registration Summary:\n";
        $response .= "Name: " . $data['main_life']['full_name'] . "\n";
        $response .= "Product: " . $selected_product['product_name'] . "\n";
        $response .= "Sum Assured: GHS " . number_format($selected_sum_assured, 2) . "\n";
        $response .= "Monthly Premium: GHS " . number_format($premium, 2) . "\n";
        
        if (isset($data['has_dependents']) && $data['has_dependents']) {
            $response .= "Dependents: " . $data['num_dependents'] . "\n";
        }
        
        $response .= "\nTotal monthly payment: GHS " . number_format($premium, 2) . "\n\n";
        $response .= "1. Confirm registration\n";
        $response .= "2. Cancel";
        
        echo $response;
    } else {
        showError("Unable to calculate premium. Please try again.");
    }
}

// ============================================================================
// PREMIUM CALCULATION HANDLER
// ============================================================================

function handlePremiumCalculation($conn, $user_input, &$data) {
    // This step is handled in sum assured selection
    // Keeping the function structure for consistency
    showError("Invalid step in registration process.");
}

// ============================================================================
// FINAL REGISTRATION HANDLER
// ============================================================================

function handleFinalRegistration($conn, $user_input, &$data) {
    if ($user_input === '1') {
        // Confirm registration
        $session = $_SESSION['ussd_session'];
        $main_life = $data['main_life'];
        $selected_product = $data['selected_product'];
        $selected_sum_assured = $data['selected_sum_assured'];
        $premium = $data['premium'];
        
        // Prepare registration data for API
        $registration_data = [
            'mobile_money_number' => $session['msisdn'],
            'full_name' => $main_life['full_name'],
            'date_of_birth' => $main_life['date_of_birth'],
            'gender' => $main_life['gender'],
            'sum_assured' => $selected_sum_assured,
            'product_id' => $selected_product['product_index'],
            'product_name' => $selected_product['product_name'],
            'total_premium' => $premium,
            'momo_network' => $session['network']
        ];
        
        // Add dependents if any
        if (isset($data['has_dependents']) && $data['has_dependents'] && !empty($data['dependents'])) {
            $dependents = $data['dependents'];
            
            $registration_data['addlife_mobile_money_number'] = [];
            $registration_data['addlife_full_name'] = [];
            $registration_data['addlife_date_of_birth'] = [];
            $registration_data['addlife_gender'] = [];
            $registration_data['addlife_relationship'] = [];
            
            foreach ($dependents as $index => $dep) {
                // For demo, use main number for dependents (in real scenario, you'd collect dependent numbers)
                $registration_data['addlife_mobile_money_number'][] = $session['msisdn'];
                $registration_data['addlife_full_name'][] = $dep['full_name'];
                $registration_data['addlife_date_of_birth'][] = $dep['date_of_birth'];
                $registration_data['addlife_gender'][] = $dep['gender'];
                $registration_data['addlife_relationship'][] = 'Dependent ' . ($index + 1);
            }
        }
        
        // Call signup API
        $api_response = callApi('signup.php', $registration_data);
        
        if ($api_response && isset($api_response['ok']) && $api_response['ok'] === true) {
            // Registration successful
            $subscription_id = $api_response['subscriptions_id'] ?? 'N/A';
            $inserted_dependents = $api_response['additional_lives_inserted'] ?? 0;
            
            // Save registration to database
            saveRegistration($conn, $session, $data, $subscription_id);
            
            // Log successful registration
            logSession($conn, $session['session_id'], $session['msisdn'], $session['network'], 
                      'REGISTRATION_SUCCESS', 'Subscription ID: ' . $subscription_id);
            
            // Show success message
            $response = "END Registration successful!\n\n";
            $response .= "Thank you for choosing Abrabo Pa Insurance.\n";
            $response .= "Subscription ID: " . $subscription_id . "\n";
            $response .= "Monthly Premium: GHS " . number_format($premium, 2) . "\n";
            $response .= "Sum Assured: GHS " . number_format($selected_sum_assured, 2) . "\n";
            
            if ($inserted_dependents > 0) {
                $response .= "Dependents added: " . $inserted_dependents . "\n";
            }
            
            $response .= "\nYou will receive a confirmation SMS shortly.";
            
            echo $response;
            
            // Clear session data
            session_destroy();
            
        } else {
            // Registration failed
            $error_msg = $api_response['error'] ?? 'Unknown error occurred';
            
            // Log registration failure
            logSession($conn, $session['session_id'], $session['msisdn'], $session['network'], 
                      'REGISTRATION_FAILED', $error_msg);
            
            showError("Registration failed: " . $error_msg . "\nPlease try again or contact support.");
        }
        
    } elseif ($user_input === '2') {
        // Cancel registration
        $session = $_SESSION['ussd_session'];
        
        // Log cancellation
        logSession($conn, $session['session_id'], $session['msisdn'], $session['network'], 
                  'REGISTRATION_CANCELLED', 'User cancelled registration');
        
        echo "END Registration cancelled.\nThank you for your interest in Abrabo Pa Insurance.";
        
        // Clear session data
        session_destroy();
        
    } else {
        echo "CON Invalid selection. Please select:\n";
        echo "1. Confirm registration\n";
        echo "2. Cancel";
    }
}

// ============================================================================
// COMING SOON HANDLER
// ============================================================================

function handleComingSoon($user_input) {
    // This handles the "coming soon" responses for other menu items
    echo "END Feature coming soon!\nCurrently only registration is available.";
    session_destroy();
}

// ============================================================================
// HELPER FUNCTIONS
// ============================================================================

function createDatabaseConnection($config) {
    $conn = new mysqli(
        $config['host'],
        $config['user'],
        $config['pass'],
        $config['name'],
        $config['port']
    );
    
    if ($conn->connect_error) {
        die("Database connection failed: " . $conn->connect_error);
    }
    
    $conn->set_charset("utf8mb4");
    return $conn;
}

function createTables($conn) {
    $tables = [
        "CREATE TABLE IF NOT EXISTS ussd_sessions (
            id INT AUTO_INCREMENT PRIMARY KEY,
            session_id VARCHAR(100) NOT NULL,
            msisdn VARCHAR(20) NOT NULL,
            network VARCHAR(20),
            ussd_code VARCHAR(20),
            event_type VARCHAR(50),
            event_data TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_session_id (session_id),
            INDEX idx_msisdn (msisdn),
            INDEX idx_created_at (created_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",
        
        "CREATE TABLE IF NOT EXISTS ussd_registrations (
            id INT AUTO_INCREMENT PRIMARY KEY,
            session_id VARCHAR(100) NOT NULL,
            msisdn VARCHAR(20) NOT NULL,
            full_name VARCHAR(100) NOT NULL,
            date_of_birth DATE NOT NULL,
            gender VARCHAR(10) NOT NULL,
            product_index INT NOT NULL,
            product_name VARCHAR(100) NOT NULL,
            sum_assured DECIMAL(10,2) NOT NULL,
            premium DECIMAL(10,2) NOT NULL,
            has_dependents BOOLEAN DEFAULT FALSE,
            num_dependents INT DEFAULT 0,
            dependents_data JSON,
            subscription_id VARCHAR(50),
            status ENUM('pending', 'completed', 'failed') DEFAULT 'pending',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_msisdn (msisdn),
            INDEX idx_subscription_id (subscription_id),
            INDEX idx_status (status)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4"
    ];
    
    foreach ($tables as $sql) {
        $conn->query($sql);
    }
}

function cleanPhoneNumber($phone) {
    $phone = preg_replace('/[^0-9]/', '', $phone);
    
    if (substr($phone, 0, 3) === '233') {
        return $phone;
    }
    
    if (substr($phone, 0, 1) === '0') {
        return '233' . substr($phone, 1);
    }
    
    if (strlen($phone) === 9) {
        return '233' . $phone;
    }
    
    return $phone;
}

function validateDate($date, $format = 'Y-m-d') {
    $d = DateTime::createFromFormat($format, $date);
    return $d && $d->format($format) === $date;
}

function callApi($endpoint, $data) {
    $url = API_BASE_URL . $endpoint;
    
    $ch = curl_init($url);
    
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Accept: application/json'
    ]);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_errno($ch)) {
        error_log("API Error [$url]: " . curl_error($ch));
        curl_close($ch);
        return null;
    }
    
    curl_close($ch);
    
    if ($http_code >= 200 && $http_code < 300) {
        return json_decode($response, true);
    } else {
        error_log("API HTTP Error [$url]: $http_code - $response");
        return null;
    }
}

function logSession($conn, $session_id, $msisdn, $network, $event_type, $event_data) {
    $sql = "INSERT INTO ussd_sessions (session_id, msisdn, network, event_type, event_data) 
            VALUES (?, ?, ?, ?, ?)";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param('sssss', $session_id, $msisdn, $network, $event_type, $event_data);
    $stmt->execute();
}

function saveRegistration($conn, $session, $data, $subscription_id) {
    $main_life = $data['main_life'];
    $selected_product = $data['selected_product'];
    $selected_sum_assured = $data['selected_sum_assured'];
    $premium = $data['premium'];
    
    $has_dependents = isset($data['has_dependents']) && $data['has_dependents'] ? 1 : 0;
    $num_dependents = $data['num_dependents'] ?? 0;
    $dependents_data = isset($data['dependents']) ? json_encode($data['dependents']) : '[]';
    
    $sql = "INSERT INTO ussd_registrations (
                session_id, msisdn, full_name, date_of_birth, gender,
                product_index, product_name, sum_assured, premium,
                has_dependents, num_dependents, dependents_data,
                subscription_id, status
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'completed')";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param(
        'sssssisddiiss',
        $session['session_id'],
        $session['msisdn'],
        $main_life['full_name'],
        $main_life['date_of_birth'],
        $main_life['gender'],
        $selected_product['product_index'],
        $selected_product['product_name'],
        $selected_sum_assured,
        $premium,
        $has_dependents,
        $num_dependents,
        $dependents_data,
        $subscription_id
    );
    
    $stmt->execute();
}

function showError($message) {
    echo "END " . $message;
    session_destroy();
    exit();
}

// ============================================================================
// SESSION CLEANUP ON EXIT
// ============================================================================

register_shutdown_function(function() {
    if (isset($_SESSION['ussd_session'])) {
        // Clear session after 5 minutes of inactivity
        if (time() - $_SESSION['ussd_session']['last_activity'] > 300) {
            session_destroy();
        }
    }
});
?>